(function () {
  'use strict';

  const STORAGE_TASKS = 'strom-rustu-tasks';
  const STORAGE_STATS = 'strom-rustu-stats';
  const TASKS_PER_STAGE = 5;
  const STAGE_LABELS = [
    'Splň úkoly, aby strom vyrostl',
    'Klíček sílí – pokračuj!',
    'Mladý strom roste díky tobě',
    'Tvůj strom je silný',
    'Rozkvetlý strom – skvělá práce!',
  ];

  const MOTIVATION = [
    'Každý malý krok se počítá.',
    'Dnes jsi udělal víc než včera.',
    'Tvůj strom roste s každým úkolem.',
    'Konzistence je klíč k růstu.',
    'Malé kroky vedou k velkým cílům.',
    'Každý splněný úkol tě posouvá vpřed.',
    'Ty rozhoduješ, jak vysoký tvůj strom vyroste.',
    'Růst je proces – důležitý je směr.',
    'Děkujeme, že se staráš o svůj strom.',
    'Zítra budeš rád za to, co uděláš dnes.',
  ];

  let tasks = [];
  let stats = { totalCompleted: 0, lastCompletedDate: null, streak: 0 };
  let currentFilter = 'all';

  const el = {
    taskList: document.getElementById('taskList'),
    taskInput: document.getElementById('taskInput'),
    addTaskForm: document.getElementById('addTaskForm'),
    emptyState: document.getElementById('emptyState'),
    treeContainer: document.getElementById('treeContainer'),
    tree: document.getElementById('tree'),
    progressFill: document.getElementById('progressFill'),
    treeLabel: document.getElementById('treeLabel'),
    motivationText: document.getElementById('motivationText'),
    streak: document.getElementById('streak'),
    todayCount: document.getElementById('todayCount'),
    totalCount: document.getElementById('totalCount'),
  };

  function loadData() {
    try {
      const t = localStorage.getItem(STORAGE_TASKS);
      if (t) tasks = JSON.parse(t);
      const s = localStorage.getItem(STORAGE_STATS);
      if (s) stats = JSON.parse(s);
    } catch (_) {
      tasks = [];
      stats = { totalCompleted: 0, lastCompletedDate: null, streak: 0 };
    }
  }

  function saveTasks() {
    localStorage.setItem(STORAGE_TASKS, JSON.stringify(tasks));
  }

  function saveStats() {
    localStorage.setItem(STORAGE_STATS, JSON.stringify(stats));
  }

  function todayStr() {
    return new Date().toISOString().slice(0, 10);
  }

  function updateStreak() {
    const today = todayStr();
    if (!stats.lastCompletedDate) {
      stats.streak = 0;
      return;
    }
    const last = stats.lastCompletedDate;
    const yesterday = new Date();
    yesterday.setDate(yesterday.getDate() - 1);
    const yesterdayStr = yesterday.toISOString().slice(0, 10);
    if (last === today) {
      if (stats.streak === 0) stats.streak = 1;
    } else if (last === yesterdayStr) {
      stats.streak = (stats.streak || 0) + 1;
    } else {
      stats.streak = 1;
    }
    stats.lastCompletedDate = today;
    saveStats();
  }

  function getTodayCompletedCount() {
    const today = todayStr();
    return tasks.filter(
      (t) => t.done && t.completedAt && t.completedAt.startsWith(today)
    ).length;
  }

  function getTreeStage() {
    const n = stats.totalCompleted || 0;
    return Math.min(4, Math.floor(n / TASKS_PER_STAGE));
  }

  function getProgressPercent() {
    const n = stats.totalCompleted || 0;
    const inStage = n % TASKS_PER_STAGE;
    return (inStage / TASKS_PER_STAGE) * 100;
  }

  function renderTree() {
    const stage = getTreeStage();
    const percent = getProgressPercent();
    const stages = el.tree.querySelectorAll('.tree-stage');
    stages.forEach((s, i) => {
      s.classList.toggle('visible', Number(s.dataset.stage) === stage);
    });
    el.progressFill.style.width = percent + '%';
    el.treeLabel.textContent = STAGE_LABELS[stage];
    const progressBar = document.getElementById('progressBar');
    if (progressBar) progressBar.setAttribute('aria-valuenow', Math.round(percent));
  }

  function setMotivation() {
    const i = Math.floor(Math.random() * MOTIVATION.length);
    el.motivationText.textContent = MOTIVATION[i];
  }

  function renderStats() {
    el.streak.textContent = stats.streak || 0;
    el.todayCount.textContent = getTodayCompletedCount();
    el.totalCount.textContent = stats.totalCompleted || 0;
  }

  function filterTasks(list) {
    if (currentFilter === 'active') return list.filter((t) => !t.done);
    if (currentFilter === 'done') return list.filter((t) => t.done);
    return list;
  }

  function renderTasks() {
    const list = filterTasks(tasks);
    el.taskList.innerHTML = '';
    list.forEach((task) => {
      const li = document.createElement('li');
      li.className = 'task-item' + (task.done ? ' done' : '');
      li.dataset.id = task.id;
      li.innerHTML =
        '<button type="button" class="task-check" aria-label="Označit jako splněno">' +
        (task.done ? '✓' : '') +
        '</button>' +
        '<span class="task-text"></span>' +
        '<button type="button" class="task-delete" aria-label="Smazat úkol">×</button>';
      const textEl = li.querySelector('.task-text');
      textEl.textContent = task.text;
      const checkBtn = li.querySelector('.task-check');
      const deleteBtn = li.querySelector('.task-delete');

      checkBtn.addEventListener('click', () => toggleTask(task.id));
      deleteBtn.addEventListener('click', () => deleteTask(task.id));

      el.taskList.appendChild(li);
    });

    const hasAny = list.length > 0;
    el.emptyState.classList.toggle('hidden', hasAny);
    renderStats();
    renderTree();
  }

  function addTask(text) {
    const trimmed = (text || '').trim();
    if (!trimmed) return;
    const task = {
      id: Date.now().toString(36) + Math.random().toString(36).slice(2),
      text: trimmed,
      done: false,
      completedAt: null,
      createdAt: new Date().toISOString(),
    };
    tasks.unshift(task);
    saveTasks();
    renderTasks();
    el.taskInput.value = '';
    el.taskInput.focus();
  }

  function toggleTask(id) {
    const task = tasks.find((t) => t.id === id);
    if (!task) return;
    task.done = !task.done;
    if (task.done) {
      task.completedAt = new Date().toISOString();
      stats.totalCompleted = (stats.totalCompleted || 0) + 1;
      updateStreak();
      saveStats();
    } else {
      task.completedAt = null;
      stats.totalCompleted = Math.max(0, (stats.totalCompleted || 0) - 1);
      saveStats();
    }
    saveTasks();
    renderTasks();
    setMotivation();
  }

  function deleteTask(id) {
    tasks = tasks.filter((t) => t.id !== id);
    saveTasks();
    renderTasks();
  }

  function setFilter(filter) {
    currentFilter = filter;
    document.querySelectorAll('.filter-btn').forEach((btn) => {
      const isActive = btn.dataset.filter === filter;
      btn.classList.toggle('active', isActive);
      btn.setAttribute('aria-pressed', isActive);
    });
    renderTasks();
  }

  el.addTaskForm.addEventListener('submit', (e) => {
    e.preventDefault();
    addTask(el.taskInput.value);
  });

  document.querySelectorAll('.filter-btn').forEach((btn) => {
    btn.addEventListener('click', () => setFilter(btn.dataset.filter));
  });

  loadData();
  setMotivation();
  renderTasks();

  if ('serviceWorker' in navigator) {
    navigator.serviceWorker.register('sw.js').catch(function () {});
  }
})();
